implementation module StdMenuElementClass


//	Clean Object I/O library, version 1.0.1

//	Definition of the MenuElements class for menu elements.


import	StdList, StdMisc, StdTuple
import	StdMenuDef
import	commondef, menudefaccess, menuhandle


class MenuElements m
where
	menuElementToHandles	:: !(m .ls .ps) -> [MenuElementState .ls .ps]
	getMenuElementType		::  (m .ls .ps) -> MenuElementType

/*	Translating menu elements into the internal representation.
	Fields which values can not be determined now are filled with dummy values.
	These are the following:
	-	SubMenuHandle
		-	mSubHandle			(the handle to the sub menu)
		-	mSubOSMenuNr		(the internal system id of the sub menu)
	-	MenuItemHandle
		-	mOSMenuItem			(the handle to the item)
	-	MenuSeparatorHandle
		-	mOSMenuSeparator	(the handle to the item)
	The remaining attributes are copied.
*/
instance MenuElements (AddLS m)	| MenuElements m where
	menuElementToHandles :: !(AddLS m .ls .ps) -> [MenuElementState .ls .ps]		| MenuElements m
	menuElementToHandles {addLS,addDef}
		= [MenuElementHandleToMenuElementState 
		  (MenuExtendLSHandle {	mExtendLS	= addLS
							  ,	mExtendItems= map MenuElementStateToMenuElementHandle (menuElementToHandles addDef)
							  }
		  )]
	getMenuElementType _
		= ""

instance MenuElements (NewLS m)	| MenuElements m where
	menuElementToHandles :: !(NewLS m .ls .ps) -> [MenuElementState .ls .ps]		| MenuElements m
	menuElementToHandles {newLS,newDef}
		= [MenuElementHandleToMenuElementState
		  (MenuChangeLSHandle {	mChangeLS	= newLS
							  ,	mChangeItems= map MenuElementStateToMenuElementHandle (menuElementToHandles newDef)
							  }
		  )]
	getMenuElementType _
		= ""

instance MenuElements (ListLS m)	| MenuElements m where
	menuElementToHandles :: !(ListLS m .ls .ps) -> [MenuElementState .ls .ps]		| MenuElements m
	menuElementToHandles (ListLS m)
		= [MenuElementHandleToMenuElementState
		  (MenuListLSHandle (map MenuElementStateToMenuElementHandle (flatten (map menuElementToHandles m)))
		  )]
	getMenuElementType _
		= ""

instance MenuElements NilLS where
	menuElementToHandles :: !(NilLS .ls .ps) -> [MenuElementState .ls .ps]
	menuElementToHandles NilLS
		= [MenuElementHandleToMenuElementState (MenuListLSHandle [])]
	getMenuElementType _
		= ""

instance MenuElements ((:+:) m1 m2)	| MenuElements m1 & MenuElements m2 where
	menuElementToHandles :: !((:+:) m1 m2 .ls .ps) -> [MenuElementState .ls .ps]	| MenuElements m1 & MenuElements m2
	menuElementToHandles (m1:+:m2)
		= menuElementToHandles m1 ++ menuElementToHandles m2
	getMenuElementType _
		= ""

instance MenuElements (SubMenu m)	| MenuElements m where
	menuElementToHandles :: !(SubMenu m .ls .ps) -> [MenuElementState .ls .ps]	| MenuElements m
	menuElementToHandles (SubMenu title items atts)
		# (selectAtt,atts)	= validateSelectState atts
		  (idAtt,    atts)	= validateId          atts
		= [MenuElementHandleToMenuElementState
		  (SubMenuHandle {	mSubHandle	= OSNoMenu
						 ,	mSubMenuId	= idAtt
						 ,	mSubOSMenuNr= 0
						 ,	mSubItems	= map MenuElementStateToMenuElementHandle (menuElementToHandles items)
						 ,	mSubTitle	= title
						 ,	mSubSelect	= enabled selectAtt
				// PA--- ,	mSubLS		= True
						 ,	mSubAtts	= atts
						 }
		  )]
	getMenuElementType _
		= "SubMenu"

instance MenuElements RadioMenu where
	menuElementToHandles :: !(RadioMenu .ls .ps) -> [MenuElementState .ls .ps]
	menuElementToHandles (RadioMenu items index atts)
		# nrRadios			= length items
		  validIndex		= if (nrRadios==0) 0 (SetBetween index 1 nrRadios)
		  itemHs			= validateRadioMenuIndex validIndex (map RadioMenuItemToMenuElementHandle items)
		  (selectAtt,atts)	= validateSelectState atts
		  (idAtt,    atts)	= validateId          atts
		= [MenuElementHandleToMenuElementState
		  (RadioMenuHandle {	mRadioId	= idAtt
						   ,	mRadioIndex	= validIndex
						   ,	mRadioItems	= itemHs
						   ,	mRadioSelect= enabled selectAtt
				// PA---   ,	mRadioLS	= True
						   ,	mRadioAtts	= atts
						   }
		  )]
	getMenuElementType _
		= "RadioMenu"

instance MenuElements MenuItem where
	menuElementToHandles :: !(MenuItem .ls .ps) -> [MenuElementState .ls .ps]
	menuElementToHandles (MenuItem title atts)
		# (selectAtt,atts)	= validateSelectState atts
		  (markAtt,  atts)	= validateMarkState   atts
		  (keyAtt,   atts)	= validateShortKey    atts
		  (idAtt,    atts)	= validateId          atts
		= [MenuElementHandleToMenuElementState
		  (MenuItemHandle {	mItemId		= idAtt
						  ,	mItemKey	= keyAtt
						  ,	mItemTitle	= title
						  ,	mItemSelect	= enabled selectAtt
						  ,	mItemMark	= marked  markAtt
				// PA---  ,	mItemLS		= True
						  ,	mItemAtts	= atts
						  ,	mOSMenuItem	= OSNoMenuItem
						  }
		  )]
	getMenuElementType _
		= "MenuItem"

instance MenuElements MenuSeparator where
	menuElementToHandles :: !(MenuSeparator .ls .ps) -> [MenuElementState .ls .ps]
	menuElementToHandles (MenuSeparator atts)
		# (idAtt,_)		= validateId atts
		= [MenuElementHandleToMenuElementState 
		  (MenuSeparatorHandle {	mSepId	= idAtt
					// PA---   ,	mSepLS	= True
							   ,	mOSMenuSeparator = OSNoMenuSeparator
							   }
		  )]
	getMenuElementType _
		= "MenuSeparator"


//	Obtain the SelectState attribute from the attribute list:
validateSelectState :: ![MenuAttribute .ps] -> (!SelectState,![MenuAttribute .ps])
validateSelectState atts
	# (found,selectAtt,atts)	= Remove ismenuselectstate undef atts
	| found
	= (getmenuselectstate selectAtt,atts)
	= (Able,atts)

//	Obtain the MarkState attribute from the attribute list:
validateMarkState :: ![MenuAttribute .ps] -> (!MarkState,![MenuAttribute .ps])
validateMarkState atts
	# (found,markAtt,atts)	= Remove ismenumarkstate undef atts
	| found
	= (getmenumarkstate markAtt,atts)
	= (NoMark,atts)

//	Obtain the Id attribute from the attribute list:
validateId :: ![MenuAttribute .ps] -> (!Maybe Id,![MenuAttribute .ps])
validateId atts
	# (found,idAtt,atts)	= Remove ismenuid undef atts
	| found
	= (Just (getmenuid idAtt),atts)
	= (Nothing,atts)

//	Obtain the ShortKey attribute from the attribute list:
validateShortKey :: ![MenuAttribute .ps] -> (!Maybe Char,![MenuAttribute .ps])
validateShortKey atts
	# (hasKey,keyAtt,atts)	= Remove ismenushortkey undef atts
	| hasKey
	= (Just (getmenushortkey keyAtt),atts)
	= (Nothing,atts)

//	validateRadioMenuIndex ensures that only the element at the valid index position of the RadioMenu
//	has a check mark and all others don't.
validateRadioMenuIndex :: !Int ![MenuElementHandle .ls .ps] -> [MenuElementHandle .ls .ps]
validateRadioMenuIndex index itemHs
	= fst (StateMap (\(MenuItemHandle itemH) i->(MenuItemHandle {itemH & mItemMark=i==index},i+1)) itemHs 1)
